"use client"

import type React from "react"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import {
  Gift,
  Copy,
  Check,
  Loader2,
  ChevronDown,
  ChevronUp,
  Award,
  Users,
  Percent,
  Calendar,
  AlertCircle,
  Share2,
  QrCode,
  Wallet,
} from "lucide-react"
import { createBrowserClient } from "@/lib/supabase/client"

export function ReferralModal() {
  const [open, setOpen] = useState(false)
  const [loading, setLoading] = useState(false)
  const [success, setSuccess] = useState(false)
  const [copied, setCopied] = useState<string | null>(null)
  const [generatedCode, setGeneratedCode] = useState("")
  const [generatedUrl, setGeneratedUrl] = useState("")
  const [showRules, setShowRules] = useState(false)
  const [formData, setFormData] = useState({
    name: "",
    cpf: "",
    phone: "",
    pix_key: "",
    pix_key_type: "",
  })

  const formatCPF = (value: string) => {
    const numbers = value.replace(/\D/g, "")
    if (numbers.length <= 3) return numbers
    if (numbers.length <= 6) return `${numbers.slice(0, 3)}.${numbers.slice(3)}`
    if (numbers.length <= 9) return `${numbers.slice(0, 3)}.${numbers.slice(3, 6)}.${numbers.slice(6)}`
    return `${numbers.slice(0, 3)}.${numbers.slice(3, 6)}.${numbers.slice(6, 9)}-${numbers.slice(9, 11)}`
  }

  const formatPhone = (value: string) => {
    const numbers = value.replace(/\D/g, "")
    if (numbers.length <= 2) return numbers
    if (numbers.length <= 7) return `(${numbers.slice(0, 2)}) ${numbers.slice(2)}`
    return `(${numbers.slice(0, 2)}) ${numbers.slice(2, 7)}-${numbers.slice(7, 11)}`
  }

  const generateCode = () => {
    const prefix =
      formData.name
        .split(" ")[0]
        .slice(0, 4)
        .toUpperCase()
        .replace(/[^A-Z]/g, "X") || "REF"
    const random = Math.random().toString(36).substring(2, 6).toUpperCase()
    return `${prefix}${random}`
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)

    try {
      const supabase = createBrowserClient()
      const code = generateCode()

      const { error } = await supabase.from("referral_codes").insert({
        code,
        name: formData.name,
        cpf: formData.cpf,
        phone: formData.phone,
        pix_key: formData.pix_key,
        pix_key_type: formData.pix_key_type,
        description: `Código de indicação - ${formData.name}`,
        is_active: true,
        uses_count: 0,
      })

      if (error) throw error

      const url = `${window.location.origin}/matricula?ref=${code}`
      setGeneratedCode(code)
      setGeneratedUrl(url)
      setSuccess(true)
    } catch (error: any) {
      console.error("Erro ao criar código:", error)
      alert(`Erro ao criar código: ${error.message || "Tente novamente."}`)
    } finally {
      setLoading(false)
    }
  }

  const copyToClipboard = async (text: string, type: string) => {
    await navigator.clipboard.writeText(text)
    setCopied(type)
    setTimeout(() => setCopied(null), 2000)
  }

  const shareLink = async () => {
    if (navigator.share) {
      try {
        await navigator.share({
          title: "Indicação Uniporá",
          text: `Use meu código ${generatedCode} e ganhe 10% de desconto na taxa matrícula!`,
          url: generatedUrl,
        })
      } catch (err) {
        copyToClipboard(generatedUrl, "link")
      }
    } else {
      copyToClipboard(generatedUrl, "link")
    }
  }

  const resetForm = () => {
    setFormData({ name: "", cpf: "", phone: "", pix_key: "", pix_key_type: "" })
    setSuccess(false)
    setGeneratedCode("")
    setGeneratedUrl("")
  }

  const handleOpenChange = (isOpen: boolean) => {
    setOpen(isOpen)
    if (!isOpen) {
      resetForm()
      setShowRules(false)
    }
  }

  const benefits = [
    { icon: Award, title: "R$ 200,00", subtitle: "por matrícula", color: "text-amber-500" },
    { icon: Percent, title: "10%", subtitle: "desconto na taxa da matrícula", color: "text-emerald-500" },
    { icon: Users, title: "Ilimitado", subtitle: "indicações", color: "text-blue-500" },
  ]

  const rules = [
    "O participante deverá acessar a plataforma oficial da instituição, realizar seu cadastro e gerar um link exclusivo de indicação para o Processo Seletivo ou Curso de Pós-Graduação.",
    "Para cada candidato indicado que realizar a inscrição por meio do link de indicação, efetuar a prova, for aprovado e concretizar a matrícula mediante pagamento do boleto de matrícula, o indicador fará jus a uma premiação no valor de R$ 200,00 (duzentos reais) por aluno matriculado.",
    "Não será devida a premiação ao indicador caso o candidato indicado já tenha realizado inscrição anteriormente pelos canais oficiais da UNIPORÁ, sem a utilização de link ou código de indicação, ainda que venha a efetivar nova inscrição posteriormente utilizando o referido link, uma vez que a inscrição prévia invalida a indicação.",
    "Não será garantido o pagamento da premiação ao indicador nos casos em que o candidato indicado preencher o link de inscrição e efetivar a matrícula após o prazo de 48 (quarenta e oito) horas.",
    "Os candidatos que realizarem a inscrição por meio de link de indicação terão direito a 10% (dez por cento) de desconto na taxa da matrícula, como benefício promocional, o que constitui um atrativo para divulgação do link pelo indicador.",
    "O pagamento da premiação será realizado após o encerramento do período de matrículas, no início do mês de abril, conforme critérios administrativos da instituição.",
  ]

  const pixKeyTypes = [
    { value: "cpf", label: "CPF" },
    { value: "email", label: "E-mail" },
    { value: "telefone", label: "Telefone" },
    { value: "aleatoria", label: "Chave Aleatória" },
  ]

  const getPixKeyPlaceholder = () => {
    switch (formData.pix_key_type) {
      case "cpf":
        return "000.000.000-00"
      case "email":
        return "exemplo@email.com"
      case "telefone":
        return "+55 00 00000-0000"
      case "aleatoria":
        return "Chave aleatória"
      default:
        return "Selecione o tipo primeiro"
    }
  }

  return (
    <Dialog open={open} onOpenChange={handleOpenChange}>
      <DialogTrigger asChild>
        <Button
          variant="ghost"
          className="text-sm font-light text-amber-600 hover:text-amber-600 hover:bg-amber-500/10 gap-2"
        >
          <Gift className="h-4 w-4" />
          Indicou Ganhou
        </Button>
      </DialogTrigger>
      <DialogContent className="sm:max-w-lg max-h-[90vh] overflow-y-auto">
        <DialogHeader className="text-center pb-2">
          <div className="mx-auto w-16 h-16 bg-gradient-to-br from-amber-400 to-amber-600 rounded-full flex items-center justify-center mb-3">
            <Gift className="h-8 w-8 text-white" />
          </div>
          <DialogTitle className="text-2xl font-bold">Indicou Ganhou</DialogTitle>
          <p className="text-muted-foreground text-sm mt-1">
            Indique amigos e ganhe R$ 200,00 por cada matrícula realizada!
          </p>
        </DialogHeader>

        {/* Benefits Cards */}
        <div className="grid grid-cols-3 gap-3 my-4">
          {benefits.map((benefit, index) => (
            <div key={index} className="bg-muted/50 rounded-lg p-3 text-center border">
              <benefit.icon className={`h-6 w-6 mx-auto mb-1 ${benefit.color}`} />
              <p className="font-bold text-sm">{benefit.title}</p>
              <p className="text-xs text-muted-foreground">{benefit.subtitle}</p>
            </div>
          ))}
        </div>

        {!success ? (
          <>
            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="name">Nome Completo *</Label>
                <Input
                  id="name"
                  placeholder="Digite seu nome completo"
                  value={formData.name}
                  onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                  required
                  className="h-11"
                />
              </div>

              <div className="grid grid-cols-2 gap-3">
                <div className="space-y-2">
                  <Label htmlFor="cpf">CPF *</Label>
                  <Input
                    id="cpf"
                    placeholder="000.000.000-00"
                    value={formData.cpf}
                    onChange={(e) => setFormData({ ...formData, cpf: formatCPF(e.target.value) })}
                    maxLength={14}
                    required
                    className="h-11"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="phone">WhatsApp *</Label>
                  <Input
                    id="phone"
                    placeholder="(00) 00000-0000"
                    value={formData.phone}
                    onChange={(e) => setFormData({ ...formData, phone: formatPhone(e.target.value) })}
                    maxLength={15}
                    required
                    className="h-11"
                  />
                </div>
              </div>

              {/* PIX key section */}
              <div className="space-y-2 p-4 bg-gradient-to-r from-emerald-50 to-teal-50 dark:from-emerald-950/30 dark:to-teal-950/30 rounded-lg border border-emerald-200 dark:border-emerald-800">
                <div className="flex items-center gap-2 mb-3">
                  <Wallet className="h-5 w-5 text-emerald-600" />
                  <Label className="text-emerald-700 dark:text-emerald-300 font-medium">
                    Chave PIX para Recebimento *
                  </Label>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-5 gap-3">
  <div className="md:col-span-2 space-y-1">
    <Label
      htmlFor="pix_key_type"
      className="text-xs text-muted-foreground"
    >
      Tipo da Chave
    </Label>

    <Select
      value={formData.pix_key_type}
      onValueChange={(value) =>
        setFormData({
          ...formData,
          pix_key_type: value,
          pix_key: "",
        })
      }
      required
    >
      <SelectTrigger className="h-11 bg-white dark:bg-background">
        <SelectValue placeholder="Selecione" />
      </SelectTrigger>
      <SelectContent>
        {pixKeyTypes.map((type) => (
          <SelectItem key={type.value} value={type.value}>
            {type.label}
          </SelectItem>
        ))}
      </SelectContent>
    </Select>
  </div>

  <div className="md:col-span-3 space-y-1">
    <Label
      htmlFor="pix_key"
      className="text-xs text-muted-foreground"
    >
      Chave PIX
    </Label>

    <Input
      id="pix_key"
      placeholder={getPixKeyPlaceholder()}
      value={formData.pix_key}
      onChange={(e) =>
        setFormData({ ...formData, pix_key: e.target.value })
      }
      required
      disabled={!formData.pix_key_type}
      className="h-11 bg-white dark:bg-background"
    />
  </div>
</div>


                <p className="text-xs text-emerald-600 dark:text-emerald-400 mt-2">
                  Esta chave será utilizada para o pagamento da premiação de R$ 200,00 por matrícula.
                </p>
              </div>

              <Button
                type="submit"
                className="w-full h-12 text-base bg-gradient-to-r from-amber-500 to-amber-600 hover:from-amber-600 hover:to-amber-700"
                disabled={loading}
              >
                {loading ? (
                  <>
                    <Loader2 className="mr-2 h-5 w-5 animate-spin" />
                    Gerando código...
                  </>
                ) : (
                  <>
                    <Gift className="mr-2 h-5 w-5" />
                    Gerar Meu Código de Indicação
                  </>
                )}
              </Button>
            </form>

            {/* Collapsible Rules */}
            <div className="mt-4 border rounded-lg">
              <button
                type="button"
                onClick={() => setShowRules(!showRules)}
                className="w-full flex items-center justify-between p-3 text-sm font-medium hover:bg-muted/50 transition-colors"
              >
                <span className="flex items-center gap-2">
                  <AlertCircle className="h-4 w-4 text-muted-foreground" />
                  Regulamento do Programa
                </span>
                {showRules ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
              </button>

              {showRules && (
                <div className="p-3 pt-0 border-t">
                  <ol className="space-y-3 text-xs text-muted-foreground">
                    {rules.map((rule, index) => (
                      <li key={index} className="flex gap-2">
                        <span className="font-bold text-foreground min-w-[20px]">{index + 1}.</span>
                        <span>{rule}</span>
                      </li>
                    ))}
                  </ol>
                </div>
              )}
            </div>
          </>
        ) : (
          <div className="space-y-5">
            {/* Success Header */}
            <div className="bg-gradient-to-r from-emerald-50 to-green-50 dark:from-emerald-950/50 dark:to-green-950/50 p-5 rounded-xl border border-emerald-200 dark:border-emerald-800 text-center">
              <div className="w-14 h-14 bg-emerald-500 rounded-full flex items-center justify-center mx-auto mb-3">
                <Check className="h-8 w-8 text-white" />
              </div>
              <h3 className="text-lg font-bold text-emerald-700 dark:text-emerald-300">Código Criado com Sucesso!</h3>
              <p className="text-sm text-emerald-600 dark:text-emerald-400 mt-1">
                Compartilhe seu código e comece a ganhar
              </p>
            </div>

            {/* Code Display */}
            <div className="bg-gradient-to-r from-amber-50 to-orange-50 dark:from-amber-950/30 dark:to-orange-950/30 p-4 rounded-xl border dark:border-amber-800 border-amber-500">
              <Label className="text-xs text-amber-700 dark:text-amber-300 font-medium">Seu Código Exclusivo</Label>
              <div className="flex items-center gap-2 mt-2">
                <div className="flex-1 bg-white dark:bg-background p-3 rounded-lg font-mono text-2xl font-bold text-center border-2 dark:border-amber-700 border-border">
                  {generatedCode}
                </div>
                <Button
                  variant="outline"
                  size="icon"
                  className="h-12 w-12 border-amber-300 hover:bg-amber-100 dark:hover:bg-amber-900 bg-amber-400"
                  onClick={() => copyToClipboard(generatedCode, "code")}
                >
                  {copied === "code" ? (
                    <Check className="h-5 w-5 text-emerald-500" />
                  ) : (
                    <Copy className="h-5 w-5 text-background" />
                  )}
                </Button>
              </div>
            </div>

            {/* Link and QR Code */}
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label className="text-xs text-muted-foreground">Link de Indicação</Label>
                <div className="bg-muted p-2 rounded-lg text-xs break-all h-20 overflow-auto">{generatedUrl}</div>
                <div className="flex flex-col gap-1">
                  <Button
                    variant="outline"
                    size="sm"
                    className="flex-1 bg-transparent py-1.5"
                    onClick={() => copyToClipboard(generatedUrl, "link")}
                  >
                    {copied === "link" ? <Check className="h-4 w-4 mr-1" /> : <Copy className="h-4 w-4 mr-1" />}
                    Copiar
                  </Button>
                  <Button variant="outline" size="sm" className="flex-1 bg-transparent py-1.5" onClick={shareLink}>
                    <Share2 className="h-4 w-4 mr-1" />
                    Enviar
                  </Button>
                </div>
              </div>

              <div className="space-y-2">
                <Label className="text-xs text-muted-foreground flex items-center gap-1">
                  <QrCode className="h-3 w-3" />
                  QR Code
                </Label>
                <div className="bg-white p-2 rounded-lg border flex items-center justify-center py-7">
                  <img
                    src={`https://api.qrserver.com/v1/create-qr-code/?size=120x120&data=${encodeURIComponent(generatedUrl)}`}
                    alt="QR Code"
                    className="w-24 h-24"
                  />
                </div>
              </div>
            </div>

            {/* Info Box */}
            <div className="bg-blue-50 dark:bg-blue-950/30 border border-blue-200 dark:border-blue-800 rounded-lg p-3">
              <div className="flex gap-2">
                <Calendar className="h-5 w-5 text-blue-500 flex-shrink-0 mt-0.5" />
                <div className="text-xs">
                  <p className="font-medium text-blue-700 dark:text-blue-300">Importante</p>
                  <p className="text-blue-600 dark:text-blue-400 mt-0.5">
                    O indicado deve realizar a matrícula em até 48 horas após a inscrição. O pagamento da premiação será
                    realizado no início do mês de abril.
                  </p>
                </div>
              </div>
            </div>

            {/* Actions */}
            <div className="flex gap-3 pt-2">
              <Button variant="outline" className="flex-1 bg-transparent" onClick={resetForm}>
                Criar Novo Código
              </Button>
              <Button
                className="flex-1 bg-gradient-to-r from-amber-500 to-amber-600 hover:from-amber-600 hover:to-amber-700"
                onClick={() => setOpen(false)}
              >
                Concluir
              </Button>
            </div>
          </div>
        )}
      </DialogContent>
    </Dialog>
  )
}
