"use client"

import type React from "react"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Textarea } from "@/components/ui/textarea"
import { Label } from "@/components/ui/label"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { UserPlus, Send } from "lucide-react"
import { createBrowserClient } from "@/lib/supabase/client"
import { useToast } from "@/hooks/use-toast"

export function ReferFriendsSection() {
  const { toast } = useToast()
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [formData, setFormData] = useState({
    your_name: "",
    your_email: "",
    your_phone: "",
    friend_name: "",
    friend_email: "",
    friend_phone: "",
    message: "",
  })

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    setFormData((prev) => ({ ...prev, [e.target.name]: e.target.value }))
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsSubmitting(true)

    try {
      const supabase = createBrowserClient()
      const { error } = await supabase.from("referrals").insert([formData])

      if (error) throw error

      toast({
        title: "Indicação enviada!",
        description: "Obrigado por indicar um amigo para a Uniporá.",
      })

      setFormData({
        your_name: "",
        your_email: "",
        your_phone: "",
        friend_name: "",
        friend_email: "",
        friend_phone: "",
        message: "",
      })
    } catch (error) {
      toast({
        title: "Erro ao enviar",
        description: "Ocorreu um erro ao enviar sua indicação. Tente novamente.",
        variant: "destructive",
      })
    } finally {
      setIsSubmitting(false)
    }
  }

  return (
    <section id="indique-amigos" className="py-20 bg-muted/30">
      <div className="container mx-auto px-4">
        <div className="max-w-4xl mx-auto">
          <div className="text-center mb-12">
            <UserPlus className="h-12 w-12 text-primary mx-auto mb-4" />
            <h2 className="text-3xl md:text-4xl font-bold mb-4">Indique Amigos</h2>
            <p className="text-lg text-muted-foreground">
              Conhece alguém que gostaria de estudar na Uniporá? Indique e ajude a transformar vidas!
            </p>
          </div>

          <Card>
            <CardHeader>
              <CardTitle>Formulário de Indicação</CardTitle>
              <CardDescription>Preencha os dados abaixo para indicar um amigo</CardDescription>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleSubmit} className="space-y-6">
                <div className="space-y-4">
                  <h3 className="font-semibold text-lg">Seus Dados</h3>
                  <div className="grid md:grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="your_name">Seu Nome *</Label>
                      <Input
                        id="your_name"
                        name="your_name"
                        value={formData.your_name}
                        onChange={handleChange}
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="your_email">Seu E-mail *</Label>
                      <Input
                        id="your_email"
                        name="your_email"
                        type="email"
                        value={formData.your_email}
                        onChange={handleChange}
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="your_phone">Seu Telefone *</Label>
                      <Input
                        id="your_phone"
                        name="your_phone"
                        type="tel"
                        value={formData.your_phone}
                        onChange={handleChange}
                        required
                      />
                    </div>
                  </div>
                </div>

                <div className="space-y-4">
                  <h3 className="font-semibold text-lg">Dados do Amigo</h3>
                  <div className="grid md:grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="friend_name">Nome do Amigo *</Label>
                      <Input
                        id="friend_name"
                        name="friend_name"
                        value={formData.friend_name}
                        onChange={handleChange}
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="friend_email">E-mail do Amigo *</Label>
                      <Input
                        id="friend_email"
                        name="friend_email"
                        type="email"
                        value={formData.friend_email}
                        onChange={handleChange}
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="friend_phone">Telefone do Amigo *</Label>
                      <Input
                        id="friend_phone"
                        name="friend_phone"
                        type="tel"
                        value={formData.friend_phone}
                        onChange={handleChange}
                        required
                      />
                    </div>
                  </div>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="message">Mensagem (Opcional)</Label>
                  <Textarea
                    id="message"
                    name="message"
                    value={formData.message}
                    onChange={handleChange}
                    placeholder="Conte-nos mais sobre seu amigo e por que ele seria um ótimo aluno..."
                    rows={4}
                  />
                </div>

                <Button type="submit" className="w-full" disabled={isSubmitting}>
                  <Send className="h-4 w-4 mr-2" />
                  {isSubmitting ? "Enviando..." : "Enviar Indicação"}
                </Button>
              </form>
            </CardContent>
          </Card>
        </div>
      </div>
    </section>
  )
}
