"use client"

import { MapPin, Phone, Mail, Clock } from "lucide-react"
import { Card } from "@/components/ui/card"
import { useEffect, useState } from "react"
import { createClient } from "@/lib/supabase/client"

export function ContactInfo() {
  const [settings, setSettings] = useState<{ [key: string]: string }>({})
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    loadSettings()
  }, [])

  const loadSettings = async () => {
    const supabase = createClient()
    const { data } = await supabase.from("site_settings").select("*")

    if (data) {
      const settingsObj: { [key: string]: string } = {}
      data.forEach((item) => {
        settingsObj[item.key] = item.value
      })
      setSettings(settingsObj)
    }
    setIsLoading(false)
  }

  if (isLoading) {
    return <div className="text-center py-8">Carregando informações...</div>
  }

  const contactDetails = [
    {
      icon: MapPin,
      title: "Endereço",
      content: settings.address || "Iporá, Goiás, Brasil",
      description: "Campus Central",
    },
    {
      icon: Phone,
      title: "Telefone / WhatsApp",
      content: settings.phone || "(64) 3674-1234",
      description: "Atendimento rápido",
    },
    {
      icon: Mail,
      title: "E-mail",
      content: settings.email || "contato@unipora.edu.br",
      description: "Resposta em até 24h",
    },
    {
      icon: Clock,
      title: "Horário de Atendimento",
      content: "Segunda a Sexta",
      description: "8h às 18h",
    },
  ]

  return (
    <div className="space-y-6">
      <div>
        <h2 className="text-3xl font-bold mb-4">Informações de contato</h2>
        <p className="text-muted-foreground text-pretty">
          Nossa equipe está pronta para atendê-lo. Escolha a melhor forma de entrar em contato.
        </p>
      </div>

      <div className="space-y-4">
        {contactDetails.map((detail, index) => {
          const IconComponent = detail.icon
          return (
            <Card
              key={index}
              className="p-5 hover:shadow-lg transition-all duration-300 hover:-translate-y-0.5 border-2 animate-fade-in-up"
              style={{ animationDelay: `${index * 100}ms` }}
            >
              <div className="flex items-start gap-4">
                <div className="w-12 h-12 rounded-xl bg-primary/10 flex items-center justify-center flex-shrink-0">
                  <IconComponent className="h-6 w-6 text-primary" />
                </div>
                <div className="flex-1">
                  <h3 className="font-semibold mb-1">{detail.title}</h3>
                  <p className="text-foreground font-medium mb-0.5">{detail.content}</p>
                  <p className="text-sm text-muted-foreground">{detail.description}</p>
                </div>
              </div>
            </Card>
          )
        })}
      </div>

      <Card className="overflow-hidden border-2">
        <div className="aspect-video bg-muted relative">
          <iframe
            src={
              settings.maps_embed_url ||
              "https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d3847.2!2d-51.1167!3d-16.4432!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x0%3A0x0!2zMTbCsDI2JzM1LjUiUyA1McKwMDcnMDAuMSJX!5e0!3m2!1spt-BR!2sbr!4v1234567890"
            }
            width="100%"
            height="100%"
            style={{ border: 0 }}
            allowFullScreen
            loading="lazy"
            referrerPolicy="no-referrer-when-downgrade"
            title="Localização Uniporá"
            className="absolute inset-0"
          />
        </div>
      </Card>
    </div>
  )
}
